package skyprocstarter;

import skyproc.ARMO;
import skyproc.COBJ;
import skyproc.Condition;
import skyproc.Condition.Operator;
import skyproc.Condition.P_FormID;
import skyproc.Condition.RunOnType;
import skyproc.FormID;
import skyproc.Mod;
import skyproc.WEAP;

public final class Recipe {
    /** Prevent construction of this class. */
    private Recipe() { new AssertionError(); }
    
    ///////////////////////////////////////////////////////////////////////////////////////////////
    ///////////////////////////////////////////////////////////////////////////////////////////////
    //                    METHODS                      ////////////////////////////////////////////
    ///////////////////////////////////////////////////////////////////////////////////////////////
    ///////////////////////////////////////////////////////////////////////////////////////////////
    
    /**
     * Creates an etched weapon recipe.
     * @param patch the {@link Mod} where the recipe should be placed
     * @param weapon the {@link WEAP} weapon to be etched
     * @param materialType the integer material type of the etched item, from {@link Etch}
     * @param newWeapon the {@link WEAP} etched weapon
     */
    public static void makeWeapon(Mod patch, WEAP weapon, int materialType, WEAP newWeapon) {
        COBJ recipe = makeItem(weapon.getForm(), materialType, 1, newWeapon.getForm(),
                newWeapon.getEDID(), Forms.BENCH_FORGE);
        if (recipe != null) {
            patch.addRecord(recipe);
        }
        
        int bestMaterialType = Etch.getBestMaterialType(newWeapon.getKeywordSet().getKeywordRefs());
        COBJ temper = temperItem(newWeapon.getForm(), bestMaterialType, newWeapon.getEDID(),
                Forms.BENCH_GRINDSTONE);
        if (temper != null) {
            patch.addRecord(temper);
        }
    }
    
    /**
     * Creates an etched armor recipe.
     * @param patch the {@link Mod} where the recipe should be placed
     * @param armor the {@link ARMO} armor to be etched
     * @param materialType the integer material type of the etched item, from {@link Etch}
     * @param newArmor the {@link ARMO} etched armor
     */
    public static void makeArmor(Mod patch, ARMO armor, int materialType, ARMO newArmor) {
        COBJ recipe = makeItem(armor.getForm(), materialType, 2, newArmor.getForm(), 
                newArmor.getEDID(), Forms.BENCH_FORGE);
        if (recipe != null) {
            patch.addRecord(recipe);
        }
        
        int bestMaterialType = Etch.getBestMaterialType(newArmor.getKeywordSet().getKeywordRefs());
        COBJ temper = temperItem(newArmor.getForm(), bestMaterialType, newArmor.getEDID(),
                Forms.BENCH_TABLE);
        if (temper != null) {
            patch.addRecord(temper);
        }
    }
    
    /**
     * Creates a recipe for etching an item.
     * @param oldItem the {@link FormID} of the item to be etched
     * @param materialType the integer material type of the etched item, from {@link Etch}
     * @param quantity the integer quantity of the reinforcement item
     * @param newItem the {@link FormID} of the etched item
     * @param newItemEDID the String editor ID of the etched item
     * @param bench the {@link FormID} where the recipe should appear
     * @return the {@link COBJ} etching recipe, or {@code null} if no recipe was created
     */
    private static COBJ makeItem(FormID oldItem, int materialType, int quantity, FormID newItem, 
            String newItemEDID, FormID bench) {
        COBJ recipe = new COBJ("EtchRecipe" + newItemEDID);
        switch (materialType) {
            case Etch.TO_SILVER:        configureRecipe(recipe, Forms.INGOT_SILVER, quantity, 
                                                oldItem, Forms.PERK_STEEL); break;
            
            case Etch.TO_ORCISH:        configureRecipe(recipe, Forms.INGOT_ORICHALCUM, quantity, 
                                                oldItem, Forms.PERK_ORCISH); break;
            
            case Etch.TO_GLASS:         configureRecipe(recipe, Forms.INGOT_MALACHITE, quantity, 
                                                oldItem, Forms.PERK_GLASS); break;
            
            case Etch.TO_EBONY:         configureRecipe(recipe, Forms.INGOT_EBONY, quantity, 
                                                oldItem, Forms.PERK_EBONY); break;
            
            case Etch.TO_DRAGON_SCALE:  configureRecipe(recipe, Forms.DRAGON_SCALE, quantity, 
                                                oldItem, Forms.PERK_DRAGON); break;
            
            case Etch.TO_DRAGON_BONE:   configureRecipe(recipe, Forms.DRAGON_BONE, quantity, 
                                                oldItem, Forms.PERK_DRAGON); break;
            
            case Etch.TO_DAEDRIC:       configureRecipe(recipe, Forms.DAEDRA_HEART, quantity, 
                                                oldItem, Forms.PERK_DAEDRIC); break;
            
            default:                    return null;
        }
        
        recipe.setResultFormID(newItem);
        recipe.setOutputQuantity(1);
        recipe.setBenchKeywordFormID(bench);
        return recipe;
    }
    
    /**
     * Creates a recipe for tempering an item.
     * @param newItem the {@link FormID} of the etched item
     * @param bestMaterialType the integer best material type of the etched item
     * @param bench the {@link FormID} where the recipe should appear
     * @return the {@link COBJ} tempering recipe, or {@code null} if no recipe was created
     */
    private static COBJ temperItem(FormID newItem, int bestMaterialType, String newItemEDID, 
            FormID bench) {
        COBJ recipe = new COBJ("EtchTemper" + newItemEDID);
        
        
        configureRecipe(recipe, Etch.TEMPER_BEST_MATERIAL[bestMaterialType], 1, null, 
                Etch.TEMPER_BEST_PERK[bestMaterialType]);
        
        recipe.setResultFormID(newItem);
        recipe.setOutputQuantity(1);
        recipe.setBenchKeywordFormID(bench);
        return recipe;
    }
    
    /**
     * Configures the given recipe with the provided ingredients.
     * @param recipe the {@link COBJ} recipe to configure
     * @param ingredient1 the {@link FormID} of the first ingredient
     * @param quantity the integer quantity required for {@code ingredient1}
     * @param ingredient2 the {@link FormID} of the second ingredient, or {@code null} if none
     * @param perk the {@link FormID} of the required perk
     */
    private static void configureRecipe(COBJ recipe, FormID ingredient1, int quantity, 
            FormID ingredient2, FormID perk) {
        recipe.addIngredient(ingredient1, quantity);
        if (ingredient2 != null) {
            recipe.addIngredient(ingredient2, 1);
        }
        
        Condition c1 = new Condition(P_FormID.HasPerk, perk);
        c1.setOperator(Operator.EqualTo);
        c1.setRunOnType(RunOnType.Subject);
        c1.setValue(1.0f);
        recipe.addCondition(c1);
        
        if (ingredient2 != null) {
            Condition c2 = new Condition(P_FormID.GetItemCount, ingredient2);
            c2.setOperator(Operator.GreaterThanOrEqual);
            c2.setRunOnType(RunOnType.Subject);
            c2.setValue(1.0f);
            recipe.addCondition(c2);
        }
    }
}
