﻿Scriptname JaxonzEnhGrab extends Quest
{enhanced positioning via keyboard for when an item is grabbed}

import Debug
import Input
import Utility
import Math
import Game

JaxonzEnhGrabObject objPositioning

EffectShader Property shdHighlight Auto
MiscObject Property objToken Auto

Spell Property splMoveToTarget Auto
GlobalVariable Property gfTargetX Auto
GlobalVariable Property gfTargetY Auto
GlobalVariable Property gfTargetZ Auto

ReferenceAlias Property refMannequin Auto
ReferenceAlias Property refSimpleObject Auto
ReferenceAlias Property refWeaponPlaque Auto
ReferenceAlias Property refWeaponPlaqueCOA Auto
ReferenceAlias Property refDisplayCase Auto

ObjectReference Property objPlayer Auto
;ObjectReference Property objLimboMarker Auto

;ObjectReference Property objAxesMarker Auto
;ObjectReference Property objRotationMarker Auto
Activator Property actNoDriftMarker Auto

int iMotion_Dynamic = 1	;: The object will be simulated by havok (fall, etc...). It appears that the engine may automatically set this object to the SphereInertia or BoxInertia motion type.
int iMotion_Keyframed = 4	;: The object will NOT be simulated by Havok (will only move if forced to, through SetPosition() or TranslateTo())
int iSuperFast = 1000000
int iBusyCount = 0

;positioning keys
GlobalVariable Property giEnabled Auto
GlobalVariable Property giHotkey Auto
GlobalVariable property kFwd auto
GlobalVariable property kBack auto
GlobalVariable property kLeft auto
GlobalVariable property kRight auto
GlobalVariable property kUp auto
GlobalVariable property kDown auto
GlobalVariable property kPitchDown auto
GlobalVariable property kPitchUp auto
GlobalVariable property kYawLeft auto
GlobalVariable property kYawRight auto
GlobalVariable property kRollLeft auto
GlobalVariable property kRollRight auto
GlobalVariable property kLevel auto
GlobalVariable property kDrop auto
GlobalVariable property kMovetoTarget auto
GlobalVariable property kCollectStatic auto
GlobalVariable property kUndo auto
GlobalVariable property kCopy auto
GlobalVariable property kReset auto
GlobalVariable property kScaleLarger auto
GlobalVariable property kScaleSmaller auto
GlobalVariable property kDelete auto

GlobalVariable property gfMovementAmount Auto
GlobalVariable Property gfRotationAmount Auto
GlobalVariable property gfMovementSpeed Auto
GlobalVariable property gfRotationSpeed Auto
GlobalVariable property giTokenWeight Auto

FormList Property flstKnownStatics Auto
ObjectReference objTargeted

Message Property msgConfirmDelete auto

Event OnInit()
	CheckSKSEPlugin(kWarnOnFail)
	OnSettingsChanged()
EndEvent

int kNoWarning = 0
int kWarnOnFail = 1
int kAlwaysShowResult = 2
Function CheckSKSEPlugin(int iWarningLevel = 1)
;check if SKSE Console Plugin is installed and of sufficient version
	string sPluginName = "JaxonzConsolePlugIn"
	int iMinVersion = 1
	Int iVersion = SKSE.GetPluginVersion(sPluginName)

	if (iVersion <= 0)	;not installed
		if iWarningLevel != kNoWarning
			MessageBox("Warning:\nSKSE plugin " + sPluginName + ".DLL is not installed.\n" + \
				"You will not be able to use the command console to Position objects.\n" + \
				"This is normal for Steam installations.\nInstall from Nexus if you wish to have full functionality.")
		EndIf
	ElseIf iVersion < iMinVersion	;insufficient version
		if iWarningLevel != kNoWarning
			MessageBox("Warning:\nSKSE plugin " + sPluginName + ".DLL is not up to date\n" + \
				"Version " + iMinVersion + " required, but version " + iVersion + " installed.\n" + \
				"You should update for full functionality.")
		EndIf
	Else	;good enough
		if iWarningLevel == kAlwaysShowResult
			MessageBox("SKSE plugin " + sPluginName + ".DLL version " + iVersion + " is correctly installed.")
		EndIf
	EndIf
EndFunction

Event OnKeyDown(Int KeyCode)
	if !IsInMenuMode() ;only pay attention to hotkeys in game mode
		ObjectReference objGrabbed
		;see if there is an object grabbed
		If KeyCode == giHotkey.GetValueInt()

			if GetPlayerGrabbedRef()	;position the currently grabbed object
				Notification("Positioner selected grabbed item")
				objGrabbed = GetPlayerGrabbedRef()
			ElseIf GetCurrentCrosshairRef()	;position object in crosshairs
				Notification("Positioner selected active object")
				objGrabbed = GetCurrentCrosshairRef()
			ElseIf JaxonzConsolePlugIn.GetCurrentConsoleRef()
				;get the reference user selected in the console
				Notification("Positioner selected item picked in command console")
				objGrabbed = JaxonzConsolePlugIn.GetCurrentConsoleRef()
			Else	; get known static closest to where user is aiming
				;use targeted spell to place a marker at the collision
				splMoveToTarget.Cast(Game.GetPlayer())
				int iWaitCycles = 50
				While iWaitCycles
					Wait(0.1)
					iWaitCycles -= 1
					if gfTargetX.GetValue() != 0.0
						objGrabbed = Game.FindClosestReferenceOfAnyTypeInList(flstKnownStatics, gfTargetX.GetValue(), gfTargetY.GetValue(), gfTargetZ.GetValue(), 200.0)
						iWaitCycles = 0	;stop waiting for a result
						;reset the passed coordinates
						gfTargetX.SetValue(0)
						gfTargetY.SetValue(0)
						gfTargetZ.SetValue(0)
					EndIf
				EndWhile
				if objGrabbed
					Notification("Positioner selected item aimed at")
				EndIf
			EndIf
			if objGrabbed
				BeginObjectPositioning(objGrabbed)
				HighlightObject()
				if GetPlayerGrabbedRef()
					;cause player to let go of grabbed object
					TapKey(GetMappedKey("Activate"))
				EndIf
			Else
				Notification("No object selected")
			EndIf
			objGrabbed = none	; release any reference
		EndIf
	EndIf
EndEvent

Function OnSettingsChanged()
;update key mappings; called  by MCM menu
	UnregisterForAllKeys()
	;register keystroke callbacks
	RegisterForKey(giHotkey.GetValueInt())
	RegisterForKey(kfwd.GetValueInt())
	RegisterForKey(kBack.GetValueInt())
	RegisterForKey(kLeft.GetValueInt())
	RegisterForKey(kRight.GetValueInt())
	RegisterForKey(kUp.GetValueInt())
	RegisterForKey(kDown.GetValueInt())
	RegisterForKey(kPitchDown.GetValueInt())
	RegisterForKey(kPitchUp.GetValueInt())
	RegisterForKey(kYawLeft.GetValueInt())
	RegisterForKey(kYawRight.GetValueInt())
	RegisterForKey(kRollLeft.GetValueInt())
	RegisterForKey(kRollRight.GetValueInt())
	RegisterForKey(kLevel.GetValueInt())
	RegisterForKey(kDrop.GetValueInt())
	RegisterForKey(kMoveToTarget.GetValueInt())
	RegisterForKey(kCollectStatic.GetValueInt())
	RegisterForKey(kUndo.GetValueInt())
	RegisterForKey(kCopy.GetValueInt())
	RegisterForKey(kReset.GetValueInt())
	RegisterForKey(kScaleLarger.GetValueInt())
	RegisterForKey(kScaleSmaller.GetValueInt())
	RegisterForKey(kDelete.GetValueInt())

	objToken.SetWeight(giTokenWeight.GetValue())
EndFunction

Function BeginObjectPositioning(ObjectReference objRef)
	shdHighlight.Play(objRef)	;highlight while selecting

	;special item selected?
	if (refMannequin as JaxonzEnhGrabObject).Instantiate(objRef)
		objPositioning = refMannequin as JaxonzEnhGrabObject
	ElseIf (refWeaponPlaque as JaxonzEnhGrabObject).Instantiate(objRef)	;.flstValidForms.HasForm(objRef.GetBaseObject())
		objPositioning = refWeaponPlaque as JaxonzEnhGrabObject
	ElseIf (refWeaponPlaqueCOA as JaxonzEnhGrabObject).Instantiate(objRef)	;.flstValidForms.HasForm(objRef.GetBaseObject())
		objPositioning = refWeaponPlaqueCOA as JaxonzEnhGrabObject
	ElseIf (refDisplayCase as JaxonzEnhGrabObject).Instantiate(objRef)	;.flstValidForms.HasForm(objRef.GetBaseObject())
		objPositioning = refDisplayCase as JaxonzEnhGrabObject
	Elseif (refSimpleObject as JaxonzEnhGrabObject).Instantiate(objRef) ;assume simple ObjectReference
		objPositioning = refSimpleObject as JaxonzEnhGrabObject
	EndIf

	if objPositioning
        DeletePairedDriftController(objPositioning.GetRef())
	EndIf

	shdHighlight.Stop(objRef)	;end highlight while selecting
EndFunction

Function HighlightObject()
	if objPositioning
		GoToState("ObjectSelected")
		ObjectReference objRef = objPositioning.GetRef()
		objRef.SetMotionType(iMotion_Keyframed)
		shdHighlight.Play(objRef)
;		if objRef.GetDisplayname() == ""
;			Debug.Notification("$Object selected with Positioner")
;		Else
;			string strSelected = "$" + objRef.GetDisplayname() + " selected with Positioner"
;			Debug.Notification(strSelected)
;			Debug.Notification(objRef.GetDisplayname() + " selected with Positioner")
;		EndIf
	Else
		Notification("Unable to select object")
	EndIf
EndFunction

Function EndObjectPositioning(bool bLockInPlace = true, String sNotificationMessage = "")
;empty placeholder to prevent execution in this state
EndFunction

Function RestoreObjectAtLocation(ObjectReference objLocation)
	BeginObjectPositioning((objLocation as JaxonzEnhGrabStatic).objAssociatedStatics[0])
	objPositioning.Restore((objLocation as JaxonzEnhGrabStatic).objAssociatedStatics, (objLocation as JaxonzEnhGrabStatic))
	Wait(0.25)
	HighlightObject()
EndFunction

State ObjectSelected
	Event OnKeyDown(Int KeyCode)
		if !Utility.IsInMenuMode() ;only pay attention to hotkeys in game mode
			ObjectReference objRef = objPositioning.GetRef()

			;position variables
			Float fX = objRef.X
			Float fY = objRef.Y
			Float fZ = objRef.Z
			Float fAngleX = objRef.GetAngleX()
			Float fAngleY = objRef.GetAngleY()
			Float fAngleZ = objRef.GetAngleZ()

			;which way is the player facing
			Float fPlayerHeading = GetPlayer().GetAngleZ()
			;convert from game angle to trig angle
			;make sure is always a positive angle
			if ( fPlayerHeading < 90 )
	  			fPlayerHeading = 90.0 - fPlayerHeading
			else
				fPlayerHeading = 450.0 - fPlayerHeading
			endif

			;default speed and movement
			Float fRotationAmount = gfRotationAmount.GetValue()
			Float fRotationSpeed = gfRotationSpeed.GetValue()
			Float fMovementAmount = gfMovementAmount.GetValue()
			Float fMovementSpeed = gfMovementSpeed.GetValue()

			;movement keys
			If KeyCode == kFwd.GetValueInt()
				GotoState("ObjectInMotion")
				fX = fX + (fMovementAmount * cos(fPlayerHeading))
				fY = FY + (fMovementAmount * sin(fPlayerHeading))
			ElseIf KeyCode == kBack.GetValueInt()
				GotoState("ObjectInMotion")
				fX = fX - (fMovementAmount * cos(fPlayerHeading))
				fY = FY - (fMovementAmount * sin(fPlayerHeading))
			ElseIf KeyCode == kLeft.GetValueInt()
				GotoState("ObjectInMotion")
				fX = fX + (fMovementAmount * cos(fPlayerHeading + 90.0))
				fY = FY + (fMovementAmount * sin(fPlayerHeading + 90.0))
			ElseIf KeyCode == kRight.GetValueInt()
				GotoState("ObjectInMotion")
				fX = fX + (fMovementAmount * cos(fPlayerHeading - 90.0))
				fY = FY + (fMovementAmount * sin(fPlayerHeading - 90.0))
			ElseIf KeyCode == kUp.GetValueInt()
				GotoState("ObjectInMotion")
				fZ += fMovementAmount
			ElseIf KeyCode == kDown.GetValueInt()
				GotoState("ObjectInMotion")
				fZ -= fMovementAmount
			ElseIf KeyCode == kPitchUp.GetValueInt()
				GotoState("ObjectInMotion")
				fAngleY +=  fRotationAmount
			ElseIf KeyCode == kPitchDown.GetValueInt()
				GotoState("ObjectInMotion")
				fAngleY -=  fRotationAmount
			ElseIf KeyCode == kRollLeft.GetValueInt()
				GotoState("ObjectInMotion")
				fAngleX += fRotationAmount
			ElseIf KeyCode == kRollRight.GetValueInt()
				GotoState("ObjectInMotion")
				fAngleX -= fRotationAmount
			ElseIf KeyCode == kYawLeft.GetValueInt()
				GotoState("ObjectInMotion")
				fAngleZ -= fRotationAmount
			ElseIf KeyCode == kYawRight.GetValueInt()
				GotoState("ObjectInMotion")
				fAngleZ +=  fRotationAmount
			ElseIf KeyCode == kMoveToTarget.GetValueInt()
				splMoveToTarget.Cast(objPlayer)
				int iWaitCycles = 200
				While iWaitCycles
					Wait(0.1)
					iWaitCycles -= 1
					if gfTargetZ.GetValue() != 0.0
						fX = gfTargetX.GetValue()
						fY = gfTargetY.GetValue()
						fZ = gfTargetZ.GetValue()

						;reset the passed coordinates
						gfTargetX.SetValue(0)
						gfTargetY.SetValue(0)
						gfTargetZ.SetValue(0)

						;move quickly to target
						fRotationSpeed = iSuperFast
						fMovementSpeed = iSuperFast
						GotoState("ObjectInMotion")
						iWaitCycles = 0
					EndIf
				EndWhile
			ElseIf KeyCode == kUndo.GetValueInt()
				GotoState("ObjectInMotion")
                Notification("You can no longer undo object placements.")
			ElseIf KeyCode == kLevel.GetValueInt()
				GotoState("ObjectInMotion")
				fRotationSpeed = iSuperFast
				fMovementSpeed = iSuperFast
				fAngleX = 0
				fAngleY = 0
				fAngleZ = 0
			EndIf

			If GetState() == "ObjectInMotion"	;move the object
				objPositioning.TranslateTo(fX, fY, fZ, fAngleX, fAngleY, fAngleZ, fMovementSpeed, fRotationSpeed)

				;stop movement if key no longer pressed. This accounts for key taps which do not send OnKeyUp
				While IsKeyPressed(KeyCode)
					Wait(0.01)
				EndWhile
				objPositioning.StopTranslation()
				GoToState("ObjectSelected")
			EndIf

			;check for other non-movement functions
			If (KeyCode == kScaleLarger.GetValueInt())
				While IsKeyPressed(KeyCode)
					objPositioning.SetScale(objPositioning.GetScale() * 1.1)
					Wait(0.05)
				EndWhile
			ElseIf (KeyCode == kScaleSmaller.GetValueInt())
				While IsKeyPressed(KeyCode)
					objPositioning.SetScale(objPositioning.GetScale() * 0.9)
					Wait(0.05)
				EndWhile
			Elseif KeyCode == kDrop.GetValueInt()	;drop
				EndObjectPositioning(false, "Object dropped")
			ElseIf KeyCode == giHotkey.GetValueInt()	;lock in place
				EndObjectPositioning(true, "Object locked in place")
			ElseIf KeyCode == kCopy.GetValueInt()	;create a copy of simple objects
				;make sure it is a simple object
				If objPositioning as JaxonzEnhGrabSimpleObject
					ObjectReference objNewCopy = objRef.PlaceAtMe(objRef.GetBaseObject(), 1, true)
					objNewCopy.SetScale(objRef.GetScale())
					objNewCopy.SetMotionType(iMotion_Keyframed)	;prevent new object from moving
					Wait(0.1)	;wait because placeatme is latent... requires time to load object 3d
					EndObjectPositioning(true)			;release the current object
					BeginObjectPositioning(objNewCopy)	;begin positioning the new object
					HighlightObject()
					Notification("Now positioning new copied object. New object overlays original.")
				Else
					Notification("Only simple objects may be copied")
				EndIf
			ElseIf (KeyCode == kCollectStatic.GetValueInt()) ;&& ShowMessage("Are you sure you wish to move this object to inventory?", true, "Yes", "No")
				;first see if this is a normal, carryable object by trying to move it to player inventory
				int iOldItemCount = objPlayer.GetItemCount(objRef)
				objPlayer.AddItem(objRef)
				;if it didn't work
				if (objPlayer.GetItemCount(objRef) - iOldItemCount) == 0
					;create a new static token
					ObjectReference objStaticToken = objPlayer.PlaceAtMe(objToken)	;(Game.GetFormFromFile(0x020068FF, "JaxonzEnhGrab.esp"))	;(frmStaticToken.GetBaseObject())

					;change the name so that it reflects the item grabbed
					;if SkyUILib installed, allow user to set token name
;					objStaticToken.SetDisplayName(((Self as Form) as UILIB_1).ShowTextInput("Positioner: Name this object", objPositioning.GetDisplayName()))
					string strDisplayName = ((Self as Form) as UILIB_1).ShowTextInput("Positioner: Name this object", objPositioning.GetDisplayName())
					if strDisplayName != ""
						objStaticToken.SetDisplayName(strDisplayName)
					Else
						objStaticToken.SetDisplayName(objPositioning.GetDisplayName())	;set default token name
					EndIf

					;assign the static object reference
					(objStaticToken as JaxonzEnhGrabStatic).objAssociatedStatics = objPositioning.Collect()
					;move token to player inventory
					objPlayer.AddItem(objStaticToken)
				EndIf
				EndObjectPositioning()
			ElseIf (KeyCode == kReset.GetValueInt())
				;make sure it is a simple object
				If objPositioning as JaxonzEnhGrabSimpleObject
					;prompt for verification?
					objPositioning.GetRef().MoveToMyEditorLocation()
					objPositioning.GetRef().Reset()
					EndObjectPositioning(false,"Object returned to original location")
				Else
					Notification("Only simple objects may be returned")
				EndIf
			ElseIf (KeyCode == kDelete.GetValueInt())
				;prompt for confirmation
				if msgConfirmDelete.Show() == 0
					;disable the object
					objPositioning.Collect()
					EndObjectPositioning(true,"Object deleted")
				EndIf

			EndIf	;KeyCode

		EndIf	;!InMenuMode
	EndEvent

	Function EndObjectPositioning(bool bLockInPlace = true, String sNotificationMessage = "")
		ObjectReference objRef = objPositioning.GetRef()
		shdHighlight.Stop(objRef)
		Notification(sNotificationMessage)
		if !bLockInPlace
			objRef.SetMotionType(iMotion_Dynamic)
		Else
            CreatePairedDriftController(objRef)
		EndIf

		objPositioning.Release()
		objPositioning = none
		GoToState("")
	EndFunction

EndState	;ObjectSelected

State ObjectInMotion

EndState


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;; SALCJE FUNCTIONS ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

GlobalVariable Property SALCJE_LegacySelection Auto

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Creates a paired no-drift controller object for the given object (to be locked in place).   ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
Function CreatePairedDriftController(ObjectReference objRef)
    JaxonzEnhGrabNoDriftMarkerScript objNoDriftMarker = \
            objRef.PlaceAtMe(actNoDriftMarker, 1, true, false) as JaxonzEnhGrabNoDriftMarkerScript
    objNoDriftMarker.objPositioned = objRef

    Cell currObjRefCell = objRef.GetParentCell()
    string currCellObjKey = GetCellObjKey(currObjRefCell, objRef)
    string pairedDriftRefVal = \
            LZStringUtil.GetLoadIndependentFormDecAllowCustom(objNoDriftMarker.GetFormID())

    StorageUtil.SetStringValue(none, currCellObjKey, pairedDriftRefVal)
EndFunction


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Deletes the no-drift controller object that is paired with the given object, if one exists. ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
Function DeletePairedDriftController(ObjectReference currObjRef)
    Cell currObjRefCell = currObjRef.GetParentCell()
    string currCellObjKey = GetCellObjKey(currObjRefCell, currObjRef)

    ;; FAST METHOD (HASH LOOKUP)
    If StorageUtil.HasStringValue(none, currCellObjKey)
        string pairedDriftObjRefVal = StorageUtil.PluckStringValue(none, currCellObjKey)
        int pairedDriftObjID = \
                LZStringUtil.ResolveLoadIndependentFormAllowCustom(pairedDriftObjRefVal)

        If pairedDriftObjID != 0
            Form pdo = Game.GetFormEx(pairedDriftObjID)
            If pdo != None
                ObjectReference pdoar = pdo as JaxonzEnhGrabNoDriftMarkerScript
                If (pdoar as JaxonzEnhGrabNoDriftMarkerScript) \
                        && (pdoar as JaxonzEnhGrabNoDriftMarkerScript).objPositioned == currObjRef
                    pdoar.DisableNoWait()
                    pdoar.Delete()
                EndIf
            EndIf
        EndIf

        Return
    EndIf

    If SALCJE_LegacySelection.GetValue() == 0
        Return ; skip fallback method for new saves / placed objects
    EndIf

    ;; FALLBACK METHOD (PRE-MOD REFERENCES ON EXISTING SAVE)
    int i = currObjRefCell.GetNumRefs(24) ; activator
    While i
        i -= 1
        ObjectReference pdoar = currObjRefCell.GetNthRef(i, 24) as JaxonzEnhGrabNoDriftMarkerScript
        if (pdoar as JaxonzEnhGrabNoDriftMarkerScript) \
                && (pdoar as JaxonzEnhGrabNoDriftMarkerScript).objPositioned == currObjRef
            pdoar.DisableNoWait()
            pdoar.Delete()
        EndIf
    EndWhile
EndFunction


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Generates a unique key to represent the given object within its cell.                       ;;;
;;; The resulting key will be independent of any changes to load order.                         ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
string Function GetCellObjKey(Cell currObjRefCell, ObjectReference currObjRef)
    Return "SALCJE__" \
            + LZStringUtil.GetLoadIndependentFormDecAllowCustom(currObjRefCell.GetFormID()) \
            + "::" + LZStringUtil.GetLoadIndependentFormDecAllowCustom(currObjRef.GetFormID())
EndFunction
