﻿ScriptName LZDOCEffectImportCustomOutfits extends ActiveMagicEffect

;;;
;;; IMPORTED STRUCTURE (each item on a separate line)
;;; Do not add unneeded whitespace within any line, or whitespace between lines.
;;; All forms must follow the structure XXXXXX_PluginName.esm or XXXXXX_PluginName.esp,
;;;     where XXXXXX is the form ID of that NPC/item, excluding load order.
;;; The plugin name must correspond to the ESP/ESM in which the form was originally defined,
;;;     and must match case/spelling exactly.
;;; Load order of plugins does NOT matter!
;;;
;;;     <number of customized NPCs>
;;;     <npc form ID #1>
;;;         <npc #1 custom outfit ID>
;;;         <npc #1 default (vanilla) outfit ID>
;;;         <number of items in custom outfit>
;;;             <item form ID #1>
;;;             <item form ID #2>
;;;             ...
;;;             <item form ID #N>
;;;
;;;     <npc form ID #2>
;;;         <npc #2 custom outfit ID>
;;;         ...
;;;
;;;     ...
;;;
;;;     <npc form ID #N>
;;;         ...
;;;

Armor Property ArmorOfForcibleOutfitChanges Auto

;;; Imports all outfits from an external file. ;;;
Event OnEffectStart(Actor akTarget, Actor akCaster)
    string inputFile = "lzDefaultOutfitChanger/customOutfits.txt"
    If !LZIOUtil.ExistsFile(inputFile)
        Debug.Notification("customizations.txt does not exist.")
    EndIf

    If !LZIOUtil.LoadFile(inputFile, true, false, 100)
        Debug.Notification("Could not open import file.")
        Return
    EndIf


    ResetAllOutfits()

    int numOutfits = LZIOUtil.NextLine(inputFile) as int
    int index = numOutfits
    While index
        index -= 1
        ImportOutfit(inputFile)

        If index % 15 == 0
            Debug.Notification("Progress: " + ((numOutfits - index) * 100.0 / numOutfits) + "%")
        EndIf
    EndWhile


    LZIOUtil.CloseFile(inputFile)
    Debug.Notification("Custom outfits imported successfully")
EndEvent

;;; Imports a single outfit. ;;;
Function ImportOutfit(string inputFile)
    string formID = LZIOUtil.NextLine(inputFile)
    Outfit currOutfit = ReadOutfit(inputFile)
    Outfit oldOutfit = ReadOutfit(inputFile)
    int numItems = LZIOUtil.NextLine(inputFile) as int

    LeveledItem currOutfitList = (currOutfit.GetNthPart(0) as LeveledItem)
    int itemIndex = numItems
    While itemIndex
        itemIndex -= 1
        currOutfitList.AddForm(ReadOutfitPart(inputFile), 1, 1)
    EndWhile

    StorageUtil.SetFormValue(None, "LZDOC`" + formID + "`Default", oldOutfit)
    StorageUtil.SetFormValue(None, "LZDOC`" + formID + "`Custom", currOutfit)
    StorageUtil.StringListAdd(None, "LZDOC_ActorRegistrations", formID)
    StorageUtil.FormListRemove(None, "LZDOC_NextOutfitStack", currOutfit)

    Actor aRef = Game.GetFormEx(LZStringUtil.ResolveLoadIndependentForm(formID)) as Actor
    aRef.SetOutfit(currOutfit)
    aRef.AddItem(ArmorOfForcibleOutfitChanges, 1, True)
    Utility.Wait(0.1)
    aRef.RemoveItem(ArmorOfForcibleOutfitChanges, 1, True)
EndFunction

;;; Reads an outfit form. ;;;
Outfit Function ReadOutfit(string inputFile)
    string outfitForm = LZIOUtil.NextLine(inputFile)
    Return Game.GetFormEx(LZStringUtil.ResolveLoadIndependentForm(outfitForm)) as Outfit
EndFunction

;;; Reads an armor form. ;;;
Armor Function ReadOutfitPart(string inputFile)
    string armorForm = LZIOUtil.NextLine(inputFile)
    Return Game.GetFormEx(LZStringUtil.ResolveLoadIndependentForm(armorForm)) as Armor
EndFunction



;;; Resets all outfits in preparation for loading. ;;;
Function ResetAllOutfits()
    int numOutfits = StorageUtil.StringListCount(None, "LZDOC_ActorRegistrations")
    If numOutfits == 0
        Return
    EndIf


    int index = numOutfits
    While index
        index -= 1
        string formID = StorageUtil.StringListGet(None, "LZDOC_ActorRegistrations", index)
        Outfit currOutfit = (StorageUtil.GetFormValue(None, "LZDOC`" + formID + "`Custom") as Outfit)
        Outfit destOutfit = (StorageUtil.GetFormValue(None, "LZDOC`" + formID + "`Default") as Outfit)

        If currOutfit != None
            LeveledItem currOutfitList = (currOutfit.GetNthPart(0) as LeveledItem)
            currOutfitList.Revert()
            StorageUtil.FormListAdd(None, "LZDOC_NextOutfitStack", currOutfit) ; allow reuse of slot
        EndIf
        StorageUtil.UnsetFormValue(None, "LZDOC`" + formID + "`Custom")
        StorageUtil.UnsetFormValue(None, "LZDOC`" + formID + "`Default")

        Actor toReset = Game.GetFormEx(LZStringUtil.ResolveLoadIndependentForm(formID)) as Actor
        toReset.SetOutfit(destOutfit)
        toReset.AddItem(ArmorOfForcibleOutfitChanges, 1, True)
        Utility.Wait(0.1)
        toReset.RemoveItem(ArmorOfForcibleOutfitChanges, 1, True)

        If index % 15 == 0
            Debug.Notification("Reset Progress: " \
                    + ((numOutfits - index) * 100.0 / numOutfits) + "%")
        EndIf
    EndWhile


    StorageUtil.StringListClear(None, "LZDOC_ActorRegistrations")
    Debug.Notification("All existing custom outfits have been removed.")

    ;;; For regular NPCs, changing outfits and adding "0" of an arbitrary item are enough to force
    ;;; an outfit re-evaluation. However, many followers in particular will only recognize outfit
    ;;; changes when new items are *added* to the outfit (but not when existing items are removed).
    ;;; By adding an armor piece with overwhelming value and armor rating, followers are tricked
    ;;; into equipping that item, and then will revert back to their (new) outfit when the item has
    ;;; been removed.
EndFunction
