#!/usr/bin/env python3

import sys
sys.dont_write_bytecode = True

import os
import subprocess
import shutil
import binascii
from multiprocessing import Process, Lock, Manager
import time
import lib.pythonfsb5.extract_fsb as fsb2ogg


OPT_HELP = ["-h", "--help"]
OPT_CLEAR = ["-c", "--clear"]
OPT_FOLDER = ["-r", "--recursive"]
OPT_GAME = ["-g", "--extract_game"]
OPT_BANKS = ["-b", "--extract_banks"]
OPT_DISCARD = ["-d", "--discard_src"]
OPT_EXTRACT_ONLY = ["-e", "--extract_only"]
OPT_NOMOD = ["-n", "--no_mod_folder"]
OPT_THREADS = ["-t", "--threads"]


BANKS_MAIN = ["archives_win64\\game0", "archives_win64\\game1", "archives_win64\\game7", \
        "archives_win64\\game8"]
BANKS_SEA = []
BANKS_MECH = []
BANKS_AIR = []

FSB_MAIN = ["archives_win64\\game52", "archives_win64\\game53", "archives_win64\\game54", \
        "archives_win64\\game55", "archives_win64\\game56", "archives_win64\\game57", \
        "archives_win64\\game58", "archives_win64\\game59", "archives_win64\\game60", \
        "archives_win64\\game61", "archives_win64\\game62", "archives_win64\\game63", \
        "archives_win64\\game64", "archives_win64\\game65", "archives_win64\\game66", \
        "archives_win64\\game67", "archives_win64\\game68", "archives_win64\\game69", \
        "archives_win64\\game70", "archives_win64\\game71"]
FSB_SEA = ["dlc_win64\\bavarium_sea_heist\\arc\\game2"]
FSB_MECH = ["dlc_win64\\mech_land_assault\\arc\\game2", "dlc_win64\\mech_land_assault\\arc\\game3"]
FSB_AIR = ["dlc_win64\\sky_fortress\\game1"]

ROOT_MAIN = "main"
ROOT_SEA = "bavarium_sea_heist"
ROOT_MECH = "mech_land_assault"
ROOT_AIR = "sky_fortress"


HEADER_SIZE = 16
HEADER_DELIM = "_____"


LIB_PATH = "lib"
RUN_EXTRACT = "runExtract.bat"
RUN_TO_FSB = "runToFSB.bat"


EXT_ARCHIVE = ".tab"
EXT_ARCHIVE_ALT = ".arc"
EXT_FSB = ".fsb5"
EXT_WAV = ".wavc"
EXT_BANK = ".bank"
EXT_OGG = ".ogg"
EXT_HEADER = "header.dat"


MODE_SINGLE = 0
MODE_FOLDER = 1
MODE_GAME = 2


PATH_SRC = "\\src"
PATH_CONVERTED = "\\converted"
PATH_MOD = "__placeChangesHere"
PATH_HEADERS = "\\" + EXT_HEADER
FILE_HEADER = "_" + EXT_HEADER


'''
Entry point for the script.
Parses all script arguments and runs the extraction process.
Aborts early if any errors are detected (nonexistent files, illegal options, and so on).
'''
def processArgs():
    if len(sys.argv) < 2:
        printHelp()
        return

    mode = MODE_SINGLE
    extractBanks = False
    discardAfter = False
    convertAudio = True
    input = None
    outputDir = None
    clear = False
    makeModDirs = True
    maxThreads = 8

    i = 1
    while i < len(sys.argv):
        if sys.argv[i] == OPT_HELP[0] or sys.argv[i] == OPT_HELP[1]:
            printHelp()
            return
        elif input == None and (sys.argv[i] == OPT_CLEAR[0] or sys.argv[i] == OPT_CLEAR[1]):
            clear = True
        elif input == None and (sys.argv[i] == OPT_FOLDER[0] or sys.argv[i] == OPT_FOLDER[1]):
            mode = MODE_FOLDER
        elif input == None and (sys.argv[i] == OPT_GAME[0] or sys.argv[i] == OPT_GAME[1]):
            mode = MODE_GAME
        elif input == None and (sys.argv[i] == OPT_BANKS[0] or sys.argv[i] == OPT_BANKS[1]):
            extractBanks = True
        elif input == None and (sys.argv[i] == OPT_DISCARD[0] or sys.argv[i] == OPT_DISCARD[1]):
            discardAfter = True
        elif input == None and (sys.argv[i] == OPT_NOMOD[0] or sys.argv[i] == OPT_NOMOD[1]):
            makeModDirs = False
        elif input == None and (sys.argv[i] == OPT_THREADS[0] or sys.argv[i] == OPT_THREADS[1]):
            if i + 1 < len(sys.argv) and sys.argv[i + 1].isdigit():
                maxThreads = max(int(sys.argv[i + 1]), 1)
                i += 1
            else:
                print("illegal thread count; this option will be ignored")
        elif input == None and \
                (sys.argv[i] == OPT_EXTRACT_ONLY[0] or sys.argv[i] == OPT_EXTRACT_ONLY[1]):
            convertAudio = False
        elif input == None:
            input = sys.argv[i]
        elif outputDir == None:
            outputDir = sys.argv[i]
        else:
            print("\n")
            print("ERROR: did not expect '" + sys.argv[i] + "' at this time.")
            print("\n")
            print("Run with " + OPT_HELP[0] + " or " + OPT_HELP[1] + " to see available options.")
            return

        i += 1

    if input == None:
        print("\n")
        print("ERROR: no input file was given")
        print("\n")
        print("Run with " + OPT_HELP[0] + " or " + OPT_HELP[1] + " to see available options.")
        return

    if mode == MODE_SINGLE:
        inputTest = input.lower()
        if inputTest.endswith(EXT_ARCHIVE_ALT):
            input = input[:-len(EXT_ARCHIVE_ALT)] + EXT_ARCHIVE
        elif not inputTest.endswith((EXT_ARCHIVE, EXT_FSB, EXT_WAV)):
            print("\n")
            print("ERROR: '" + input + "' is not a supported file.")
            print("\n")
            print("If you meant to extract a folder, use " + OPT_FOLDER[0] + \
                    "; to extract an entire game, use " + OPT_GAME[0] + ".")
            print("Otherwise, the supported file types are:")
            print("    " + EXT_ARCHIVE_ALT + ", " + EXT_ARCHIVE + ", " + EXT_FSB + ", " + EXT_WAV)
            return

        if not os.path.exists(input) or os.path.isdir(input):
            print("\n")
            print("ERROR: '" + input + "' is a folder or does not exist.")
            print("\n")
            print("Please make sure you have access to this path, then try again. If your path")
            print("contains a space, the entire path must be enclosed in quotes. Unless the " + \
                    OPT_FOLDER[0] + " or")
            print(OPT_GAME[0] + " options are given, a single file is required; not a folder.")
            return

        if outputDir == None and inputTest.endswith((EXT_FSB, EXT_WAV)):
            outputDir = input + EXT_OGG
        elif inputTest.endswith((EXT_FSB, EXT_WAV)) and not outputDir.lower().endswith(EXT_OGG):
            outputDir = outputDir + EXT_OGG

    elif not os.path.isdir(input):
        print("\n")
        print("ERROR: '" + input + "' is not a directory or does not exist.")
        print("\n")
        print("If you meant to extract a file, remove the " + OPT_FOLDER[0] + " or " + \
                OPT_GAME[0] + " options if present.")
        print("Otherwise, please make sure you have access to this path, then try again.")
        print("\n")
        print("If your path contains a space, the entire path must be enclosed in quotes. If")
        print("your path is not fully-qualified e.g. with C:\\, it must be relative to the")
        print("current working directory (this may NOT be the same as the script directory).")
        return

    if outputDir == None:
        outputDir = os.path.abspath(os.getcwd()) + "\\out"

    if os.path.exists(outputDir) and not os.path.isdir(outputDir) \
            and not input.lower().endswith((EXT_FSB, EXT_WAV)):
        print("\n")
        print("ERROR: output '" + outputDir + "' is a file.")
        print("\n")
        print("The output must be a folder if either " + OPT_FOLDER[0] + " or " + OPT_GAME[0] + \
                " are present. Otherwise, the")
        print("output can only be a file if the input is a " + EXT_FSB + " or " + EXT_WAV + ".")
        print("\n")
        print("Please choose a folder or change the script options as needed, then try again.")
        return

    if os.path.exists(outputDir) and os.path.isdir(outputDir) \
            and input.lower().endswith((EXT_FSB, EXT_WAV)):
        print("\n")
        print("ERROR: output '" + outputDir + "' is a folder.")
        print("\n")
        print("The output must be a file when converting a non-archive file. Please choose a")
        print("file or change the script options as needed, then try again.")
        return

    timeStart = time.time()
    isSingleNonArchive = input.lower().endswith((EXT_FSB, EXT_WAV))
    start(input, outputDir, clear, mode, extractBanks, discardAfter, convertAudio, \
            makeModDirs, maxThreads, isSingleNonArchive)
    timeEnd = time.time()

    print("\n")
    print("================================================================================")
    if isSingleNonArchive:
        print("All done! Your extracted file is located at:")
        print("\n")
        print(os.path.abspath(outputDir))
        print(os.path.abspath(outputDir) + FILE_HEADER)
        print("\n")
        print("The " + FILE_HEADER + \
                " file contains the binary header for inserting the audio file")
        print("into your game. DO NOT DELETE, RENAME, OR MOVE THIS FILE! You will need it when")
        print("creating audio mods.")
    else:
        print("All done! Your extracted files are located at:")
        print("\n")
        print(os.path.abspath(outputDir))
        print("\n")
        print("    " + PATH_CONVERTED + \
                "\\            contains the playable audio files converted by this")
        print("                           tool.")
        print("\n")
        print("    " + PATH_SRC + "\\                  contains extracted files (unless " + \
                OPT_DISCARD[0] + " was given),")
        print("                           including any fmod banks. If you want to convert")
        print("                           banks, either copy the bank files to lib\\fmod\\bank")
        print("                           or open Fmod Bank Tools and point that tool to the")
        print("                           " + PATH_SRC + "\\sound\\fmod_banks folder.")
        print("\n")
        print("    \\" + PATH_MOD + "\\   is where you should place any modified audio files")
        print("                           prior to running compile.py. This folder will be")
        print("                           missing if you gave " + OPT_NOMOD[0] + \
                " when running this tool.")
        print("\n")
        print("DO NOT MODIFY ANY " + EXT_HEADER + \
                " FILES you may find! These are needed to compile any")
        print("changes to your audio files. If you rename or move header files, you will not be")
        print("able to put your changes back into the game.")

    print("\n")
    print("The process completed in " + str(timeEnd - timeStart) + " seconds.")
    print("================================================================================")
    print("\n")


'''
Displays the usage information for this script.
Called only by processArgs().
'''
def printHelp():
    print("\n")
    print("================================================================================")
    print("Lord Zapharos Audio Extractor for Just Cause 3")
    print("================================================================================")
    print("\n")
    print("This utility extracts and converts sound archives for Just Cause 3 (and possibly")
    print("other Avalanche games) into playable audio files. Converted files will be in OGG")
    print("format (you can play them with VLC or edit them with Audacity).")
    print("\n")
    print("You do NOT require the DLC to use this tool. When performing a full extraction,")
    print("any DLC audio will be placed in separate folders, and the main game audio will")
    print("be placed in a folder called '" + ROOT_MAIN + "'.")
    print("\n")
    print("USAGE: extract.py [options] path\\to\\file\\folder\\archive\\or\\game [outputFolder]")
    print("\n")
    print("If an output folder is not given, all files will be extracted to the working ")
    print("directory. The path\\to\\file\\folder\\archive\\or\\game argument is REQUIRED and must")
    print("either be a (1) path to a folder, if " + OPT_FOLDER[0] + \
            " is given; (2) path to the game folder,")
    print("if " + OPT_GAME[0] + \
            " is given, or (3) path to a single audio file or archive file, if neither")
    print(OPT_FOLDER[0] + " nor " + OPT_GAME[0] + \
        " are given. The output MUST be a file if neither " + OPT_FOLDER[0] + " nor " + \
        OPT_GAME[0] + " are given")
    print("and the input file is not an archive. In this case, if the output is omitted,")
    print("the converted file will be placed next to the input.")
    print("\n")
    print("Although this utility can extract fmod bank files (non-language audio e.g. music")
    print("and sound effects), Fmod Bank Tools does not have a CLI. If " + OPT_BANKS[0] + \
            " is given, you")
    print("must convert these audio files yourself by opening lib/fmod/Fmod Bank Tools.exe")
    print("and selecting the extracted banks. The output format for banks will be WAV.")
    print("\n")
    print("OPTIONS:")
    print("\n")
    print(OPT_HELP[0] + ", " + OPT_HELP[1] + \
            "              shows this message and exits.")
    print("\n")
    print(OPT_CLEAR[0] + ", " + OPT_CLEAR[1] + \
            "             deletes the previous output folder (if present) before")
    print("                        starting the extraction. By omitting this option, any")
    print("                        prior extraction can be resumed from where it left off.")
    print("                        When converting a single (non-archive) file, an existing")
    print("                        output will not be overwritten unless this option is")
    print("                        present.")
    print("\n")
    print(OPT_FOLDER[0] + ", " + OPT_FOLDER[1] + \
            "         folder mode; extracts audio from ALL loose files in the")
    print("                        provided folder path (does not extract archive files).")
    print("\n")
    print(OPT_GAME[0] + ", " + OPT_GAME[1] + \
            "      game mode; extracts audio from ALL game archive files;")
    print("                        you must provide the path to the root game directory.")
    print("\n")
    print(OPT_BANKS[0] + ", " + OPT_BANKS[1] + \
            "     valid only with " + OPT_GAME[0] + " or when given a single archive file;")
    print("                        extracts fmod banks (you must move these banks to the")
    print("                        lib/fmod/bank directory yourself to convert them into")
    print("                        playable audio files).")
    print("\n")
    print(OPT_DISCARD[0] + ", " + OPT_DISCARD[1] + \
            "       valid only with " + OPT_GAME[0] + " or when given a single archive file;")
    print("                        discards the archive's extracted files, keeping only the")
    print("                        converted/playable audio files; banks are retained if " + \
            OPT_BANKS[0])
    print("                        was given.")
    print("\n")
    print(OPT_EXTRACT_ONLY[0] + ", " + OPT_EXTRACT_ONLY[1] + \
            "      valid only with " + OPT_GAME[0] + " or when given a single archive file;")
    print("                        extracts an archive's audio files and does nothing else.")
    print("\n")
    print(OPT_NOMOD[0] + ", " + OPT_NOMOD[1] + \
            "     prevents the creation of mod folders after conversion is")
    print("                        done. This option only applies when converting archives")
    print("                        or games (not just extracting). If enabled, you will NOT")
    print("                        be able to repack modified audio files until you run")
    print("                        createModFolders.py.")
    print("\n")
    print(OPT_THREADS[0] + ", " + OPT_THREADS[1] + \
            "           valid only with " + OPT_FOLDER[0] + " or " + OPT_GAME[0] + \
            "; sets how many threads should")
    print("                        be used when converting audio files. A larger value will")
    print("                        increase conversion speed, but demands more CPU power. A")
    print("                        value between 2-16 is recommended; the default is 8. A")
    print("                        good rule of thumb is twice your CPU's core count.")
    print("\n")
    print("EXAMPLES:")
    print("\n")
    print("extract.py -c __UNKNOWN\\fsb5\\000D2DFB.fsb5     converts a single file 000D2DFB")
    print("                                               to a playable format, placing it")
    print("                                               in the same folder location and")
    print("                                               overwriting any existing file.")
    print("\n")
    print("extract.py -t 16 game52.arc                    extracts game52 from the current")
    print("                                               working directory and converts")
    print("                                               all audio files. Uses 16 threads.")
    print("\n")
    print("extract.py -t 16 -b -g \"C:\\Just Cause 3\"       performs a full game extraction,")
    print("                                               retaining any previous extracted")
    print("                                               files, and also extracts bank")
    print("                                               (non-dialogue) audio files. Use")
    print("                                               Fmod Bank Tools to convert the")
    print("                                               bank files after extraction.")
    print("\n")
    print("extract.py -r C:\\myFolder                      converts all supported files in")
    print("                                               myFolder to playable audio files.")
    print("\n")
    print("================================================================================")
    print("WARNING: AUDIO EXTRACTION AND CONVERSION MAY TAKE A **LONG** TIME! Depending on")
    print("your CPU and disk speeds, a full game extraction may take one or two HOURS to")
    print("complete. Even high-end machines often require at least 30 minutes.")
    print("\n")
    print("For a full extraction, you must have at least 23 GB of free space. However, if")
    print(OPT_DISCARD[0] + \
            " is given, only 11 GB of free space should be needed. The full game includes")
    print("over 135,000 audio files; archives may place > 14,000 files in a single folder.")
    print("\n")
    print("Please be patient, and have fun!")
    print("================================================================================")
    print("\n")


'''
Initiates the extraction and conversion process.
When extracting a full game, archives will be examined one-at-a-time to reduce disk space.
Called only by processArgs().
'''
def start(input, outputDir, clear, mode, extractBanks, discardAfter, convertAudio, \
        makeModDirs, maxThreads, isSingleNonArchive):
    print("\n")
    if isSingleNonArchive:
        convertSingle(input, outputDir, clear)
        return

    oConv = outputDir + PATH_CONVERTED
    oSrc = outputDir + PATH_SRC
    pHead = outputDir + PATH_HEADERS

    if clear:
        clearDirectory(outputDir)

    if not os.path.exists(oConv):
        os.makedirs(oConv)
    if not os.path.exists(oSrc):
        os.makedirs(oSrc)

    if mode == MODE_GAME:
        extractGame(input, oConv, oSrc, outputDir, mode, extractBanks, discardAfter, \
                convertAudio, maxThreads)
    elif mode == MODE_SINGLE:
        extract(input, oConv, oSrc, pHead, mode, extractBanks, discardAfter, convertAudio, \
                maxThreads, "", True)
    else:
        convertTreeExecutor(oConv, input, pHead, maxThreads, "")

    if makeModDirs and convertAudio:
        import createModFolders
        if mode == MODE_GAME:
            createModFolders.start(outputDir + PATH_CONVERTED, outputDir + "\\" + PATH_MOD, True)
        else:
            createModFolders.start(outputDir, outputDir + "\\" + PATH_MOD, True)


'''
Performs a full game extraction.
Called only by start().
'''
def extractGame(input, oConv, oSrc, outputDir, mode, extractBanks, discardAfter, convertAudio, \
        maxThreads):
    print("Beginning full game extraction (this may take a VERY LONG time!)")

    if extractBanks:
        for archive in BANKS_MAIN:
            oCMod, oSMod, pHead = modOutputs(oConv, oSrc, outputDir, ROOT_MAIN, strA(archive))
            extract(input + "\\" + archive + EXT_ARCHIVE, oCMod, oSMod, pHead, mode, \
            extractBanks, discardAfter, convertAudio, maxThreads, ROOT_MAIN, True)

        for archive in BANKS_SEA:
            oCMod, oSMod, pHead = modOutputs(oConv, oSrc, outputDir, ROOT_SEA, strA(archive))
            extract(input + "\\" + archive + EXT_ARCHIVE, oCMod, oSMod, pHead, mode, \
            extractBanks, discardAfter, convertAudio, maxThreads, ROOT_SEA, False)

        for archive in BANKS_MECH:
            oCMod, oSMod, pHead = modOutputs(oConv, oSrc, outputDir, ROOT_MECH, strA(archive))
            extract(input + "\\" + archive + EXT_ARCHIVE, oCMod, oSMod, pHead, mode, \
            extractBanks, discardAfter, convertAudio, maxThreads, ROOT_MECH, False)

        for archive in BANKS_AIR:
            oCMod, oSMod, pHead = modOutputs(oConv, oSrc, outputDir, ROOT_AIR, strA(archive))
            extract(input + "\\" + archive + EXT_ARCHIVE, oCMod, oSMod, pHead, mode, \
            extractBanks, discardAfter, convertAudio, maxThreads, ROOT_AIR, False)

    for archive in FSB_MAIN:
        oCMod, oSMod, pHead = modOutputs(oConv, oSrc, outputDir, ROOT_MAIN, strA(archive))
        extract(input + "\\" + archive + EXT_ARCHIVE, oCMod, oSMod, pHead, mode, extractBanks, \
        discardAfter, convertAudio, maxThreads, ROOT_MAIN, True)

    for archive in FSB_SEA:
        oCMod, oSMod, pHead = modOutputs(oConv, oSrc, outputDir, ROOT_SEA, strA(archive))
        extract(input + "\\" + archive + EXT_ARCHIVE, oCMod, oSMod, pHead, mode, extractBanks, \
        discardAfter, convertAudio, maxThreads, ROOT_SEA, False)

    for archive in FSB_MECH:
        oCMod, oSMod, pHead = modOutputs(oConv, oSrc, outputDir, ROOT_MECH, strA(archive))
        extract(input + "\\" + archive + EXT_ARCHIVE, oCMod, oSMod, pHead, mode, extractBanks, \
        discardAfter, convertAudio, maxThreads, ROOT_MECH, False)

    for archive in FSB_AIR:
        oCMod, oSMod, pHead = modOutputs(oConv, oSrc, outputDir, ROOT_AIR, strA(archive))
        extract(input + "\\" + archive + EXT_ARCHIVE, oCMod, oSMod, pHead, mode, extractBanks, \
        discardAfter, convertAudio, maxThreads, ROOT_AIR, False)


'''
Strips an archive to its base pathname.
Called only by extractGame().
'''
def strA(archive):
    return os.path.basename(archive)

'''
Modifies the output and header file directories to reflect an arbitrary root and base archive name.
Called only by extractGame().
'''
def modOutputs(oConv, oSrc, outputDir, root, baseArchive):
    oCMod = oConv + "\\" + root + "\\" + baseArchive
    if not os.path.exists(oCMod):
        os.makedirs(oCMod)

    oSMod = oSrc + "\\" + root + "\\" + baseArchive
    if not os.path.exists(oSMod):
        os.makedirs(oSMod)

    pHead = oConv + "\\" + root + "\\" + baseArchive + FILE_HEADER

    return oCMod, oSMod, pHead


'''
Extracts files from a single archive.
Does nothing for single-file conversion or when processing a folder of loose files.
Called by start() or by extractGame().
'''
def extract(input, oConv, oSrc, pHead, mode, extractBanks, discardAfter, convertAudio, \
        maxThreads, headerPrefix, isRequired):
    if not os.path.exists(input):
        if isRequired:
            print("File '" + input + "' does not exist; skipping")
        return

    archiveBase = input[:-len(EXT_ARCHIVE)] + EXT_ARCHIVE_ALT
    if not os.path.exists(archiveBase):
        print("Archive data '" + archiveBase + "' does not exist; skipping")
        return

    print("Extracting " + input + " (this may take a while)")
    try:
        command = RUN_EXTRACT + " \"" + os.path.abspath(input) + "\" \"" + \
                os.path.abspath(oSrc) + "\""
        subprocess.check_call(command, shell=True, cwd=LIB_PATH)

    except Exception as e:
        print("\n")
        print(e)
        print("\nERROR: could not extract archive!")
        print("\n")
        quit()

    keepFiles = (EXT_FSB, EXT_WAV, EXT_BANK)
    if not extractBanks:
        keepFiles = (EXT_FSB, EXT_WAV)
    clearDirectoryFilter(oSrc, keepFiles)

    if convertAudio:
        convertTreeExecutor(oConv, oSrc, pHead, maxThreads, headerPrefix)

        if discardAfter:
            if extractBanks:
                clearDirectoryFilter(oSrc, (EXT_BANK))
            else:
                clearDirectory(oSrc)


'''
Performs a threaded conversion of files from their extracted format into a playable format.
Also responsible for creating headers for later reinsertion of modified audio streams.
Called by start() or by extract(), only if audio conversion was requested.
'''
def convertTreeExecutor(oConv, oSrc, headerFile, maxThreads, headerPrefix):
    manager = Manager()
    resultList = manager.list()
    pool = []
    lock = Lock()

    convertTree("", oConv, oSrc, pool, resultList, lock, maxThreads, headerPrefix)
    for p in pool:
        p.join()

    with open(headerFile, "a") as outfile:
        outfile.writelines(resultList)


'''
Recursively examines each file in oSrc and performs a (threaded) audio conversion.
Called only by convertTreeExecutor().
'''
def convertTree(relBase, oConv, oSrc, pool, resultList, lock, maxThreads, headerPrefix):
    oSrc = os.path.abspath(oSrc)
    canConvert = (EXT_FSB, EXT_WAV)

    for f in os.listdir(oSrc):
        filepath = os.path.join(oSrc, f)
        try:
            if (os.path.isfile(filepath) or os.path.islink(filepath)) \
                    and filepath.endswith(canConvert):
                p = Process(target=convertSingleFromTree, args=(filepath, oConv, relBase, \
                        headerPrefix, resultList, lock))
                pool.append(p)
                p.start()

                if len(pool) >= maxThreads:
                    for p in pool:
                        p.join()
                    pool.clear()

            elif os.path.isdir(filepath):
                if relBase == "":
                    relBaseNext = f
                else:
                    relBaseNext = relBase + "\\" + f
                convertTree(relBaseNext, oConv, filepath, pool, resultList, lock, maxThreads, \
                        headerPrefix)
        except Exception as e:
            print("\n")
            print(e)
            print("\nERROR: could not iterate " + oSrc)
            print("\n")
            quit()


'''
Converts a single file from convertTree() into a playable format.
Also creates the header for later reinsertion of modified audio streams.
'''
def convertSingleFromTree(input, outputRoot, prefix, headerPrefix, resultList, lock):
    outputDirs = os.path.join(outputRoot, prefix)
    try:
        if not os.path.exists(outputDirs):
            os.makedirs(outputDirs)
    except Exception:
        pass # possible race condition when creating parent folders (shared between files)

    output = os.path.join(outputDirs, os.path.basename(input))
    if output.endswith(EXT_FSB):
        output = output[:-len(EXT_FSB)] + EXT_OGG
    elif output.endswith(EXT_WAV):
        output = output[:-len(EXT_WAV)] + EXT_OGG

    hPrefix = prefix
    if len(headerPrefix) > 0:
        hPrefix = headerPrefix + "\\" + prefix
    result = decode(input, output, None, hPrefix)

    lock.acquire()
    try:
        resultList.append(result)
    finally:
        lock.release()


'''
Converts a single file into a playable format.
Also creates the header for later reinsertion of modified audio streams.
Called only by start().
'''
def convertSingle(input, outputDir, clear):
    if not clear and os.path.exists(outputDir):
        print("output file already exists; pass " + OPT_CLEAR[0] + " or " + OPT_CLEAR[1] \
                + " to overwrite")
        quit()

    headerFile = input + FILE_HEADER
    if os.path.exists(headerFile):
        os.remove(headerFile)
    decode(input, outputDir, headerFile, "")


'''
Opens a single audio file, writes/removes its header to headerFile,
and converts it to a playable audio stream.
'''
def decode(input, output, headerFile, headerPrefix):
    if os.path.exists(output):
        os.remove(output)
    headerData = trimHeader(input, output, headerFile)

    hPrefix = ""
    if len(headerPrefix) > 0:
        hPrefix = headerPrefix + "\\"
    return hPrefix + os.path.basename(input) + HEADER_DELIM + str(headerData) + "\n"


'''
Opens a single audio file and removes its game header.
The file (without header) will be written to a temporary file returned by this function.
The headerFile will be opened and a line added containing input, delimiter, and header bytes.
'''
def trimHeader(input, output, headerFile):
    header = ""

    with open(input, "rb") as infile:
        data = infile.read()
        header = binascii.hexlify(data[0:HEADER_SIZE])

        destination = os.path.abspath(output)
        oldDir = os.getcwd()
        os.chdir("lib/pythonfsb5") # library loads DLLs for which relative paths do not work
        fsb2ogg.handle_file(data[HEADER_SIZE:], destination)
        os.chdir(oldDir)

    if headerFile is not None:
        with open(headerFile, "a") as outfile:
            outfile.write(os.path.basename(input) + HEADER_DELIM + str(header) + "\n")

    return header


'''
Empties a single directory of its contents, if it exists.
'''
def clearDirectory(dir):
    if not os.path.exists(dir):
        return

    for f in os.listdir(dir):
        filepath = os.path.join(dir, f)
        try:
            if os.path.isfile(filepath) or os.path.islink(filepath):
                os.unlink(filepath)
            elif os.path.isdir(filepath):
                shutil.rmtree(filepath)
        except Exception as e:
            print("\n")
            print(e)
            print("\nERROR: could not delete contents of " + dir)
            print("\n")
            quit()


'''
Empties a single directory of any files that are missing the extensions in keepFiles.
Note that keepFiles must be a TUPLE.
'''
def clearDirectoryFilter(dir, keepFiles):
    dir = os.path.abspath(dir)
    if keepFiles == None:
        clearDirectory(dir)
        return

    if not os.path.exists(dir):
        return

    for f in os.listdir(dir):
        filepath = os.path.join(dir, f)
        try:
            if (os.path.isfile(filepath) or os.path.islink(filepath)) \
                    and not filepath.endswith(keepFiles):
                os.unlink(filepath)
            elif os.path.isdir(filepath):
                clearDirectoryFilter(filepath, keepFiles)
                if len(os.listdir(filepath)) == 0:
                    os.rmdir(filepath)
        except Exception as e:
            print("\n")
            print(e)
            print("\nERROR: could not delete contents of " + dir)
            print("\n")
            quit()


if __name__ == '__main__':
    processArgs()
