import sys
sys.dont_write_bytecode = True

import os


OPT_HELP = ["-h", "--help"]


ROOT_MAIN = "main"
ROOT_SEA = "bavarium_sea_heist"
ROOT_MECH = "mech_land_assault"
ROOT_AIR = "sky_fortress"


HEADER_DELIM = "_____"


EXT_HEADER = "header.dat"


PATH_CONVERTED = "\\converted"


'''
Entry point for the script.
Parses all script arguments and runs the creation process.
Aborts early if any errors are detected (nonexistent files, illegal options, and so on).
'''
def processArgs():
    if len(sys.argv) < 2:
        printHelp()
        return

    input = None
    output = None

    i = 1
    while i < len(sys.argv):
        if sys.argv[i] == OPT_HELP[0] or sys.argv[i] == OPT_HELP[1]:
            printHelp()
            return
        elif input == None:
            input = sys.argv[i]
        elif output == None:
            output = sys.argv[i]
        else:
            print("\n")
            print("ERROR: did not expect '" + sys.argv[i] + "' at this time.")
            print("\n")
            print("Run with " + OPT_HELP[0] + " or " + OPT_HELP[1] + " to see available options.")
            return

        i += 1

    if input == None or output == None:
        print("\n")
        print("ERROR: expected an input folder followed by an output folder.")
        return

    if not os.path.exists(input):
        print("\n")
        print("ERROR: '" + input + "' does not appear to exist.")
        print("\n")
        print("Please verify that your input folder is correct, then try again.")
        return

    if not input.endswith(PATH_CONVERTED[1:]):
        print("\n")
        print("ERROR: '" + input + "' is expected to end with: " + PATH_CONVERTED)
        print("\n")
        print("Please verify that your input folder is correct, or add " + PATH_CONVERTED + \
                " to the end of")
        print("it, then try again.")
        print("================================================================================")
        return

    start(input, output, False)

'''
Displays the usage information for this script.
Called only by processArgs().
'''
def printHelp():
    print("\n")
    print("================================================================================")
    print("Lord Zapharos Mod Folder Creator for Just Cause 3")
    print("================================================================================")
    print("\n")
    print("This utility creates a set of folders for you to place any modified audio files")
    print("that you wish to reinsert back into your game. You must have previously run")
    print("extract.py BEFORE using this tool!")
    print("\n")
    print("USAGE: createModFolders.py path\\to\\" + PATH_CONVERTED + "\\folder outputFolder")
    print("\n")
    print("The output folder will be created automatically if it does not exist. You MUST")
    print("give the " + PATH_CONVERTED + \
            " folder that was created when you previously ran extract.py.")
    print("If that folder contains " + EXT_HEADER + ", a set of folders will be created under the")
    print("outputFolder mirroring the structure in " + PATH_CONVERTED + \
            ". Otherwise, a full set of")
    print("game folders will be created e.g. " + ROOT_MAIN + ", " + ROOT_SEA + ", etc.")
    print("\n")
    print("This utility is only applicable if you extracted an archive file, folder, or the")
    print("entire game's audio files.")
    print("\n")
    print("OPTIONS:")
    print("\n")
    print(OPT_HELP[0] + ", " + OPT_HELP[1] + "              shows this message and exits.")
    print("\n")
    print("================================================================================")
    print("\n")


'''
Initiates the creation process.
Called only by processArgs() and externally by extract.py.
'''
def start(input, output, silent):
    if not silent:
        print("\n")
    inputPath = os.path.abspath(input)
    outputPath = os.path.abspath(output)

    isGame = False
    inputHeader = os.path.join(inputPath, EXT_HEADER)
    if not os.path.exists(inputHeader):
        inputHeader = os.path.join(inputPath, ROOT_MAIN)
        if not os.path.exists(inputHeader):
            print("ERROR: could not find " + EXT_HEADER + " or any game folders in: " + inputPath)
            print("\n")
            print("Please verify that your input folder is correct, then try again.")
            quit()
        else:
            isGame = True

    listHeader = []
    if isGame:
        listHeader = loadHeaderDirsGame(inputPath)
    else:
        listHeader = loadHeaderDirsFolder(inputPath)

    if len(listHeader) == 0:
        if not silent:
            print("ERROR: there were no subfolders found in: " + inputPath)
            print("\n")
            print("This error can occur if you previously converted files in a folder containing no")
            print("subfolders. In such a case, you do NOT need to use this utility.")
            print("\n")
            print("Otherwise, please make sure your input folder is correct, then try again.")
        quit()

    if not os.path.exists(outputPath):
        os.makedirs(outputPath)
    createModDirs(outputPath, listHeader)

    if silent:
        return

    print("================================================================================")
    print(str(len(listHeader)) + " folder tree(s) were created in:\n    " + inputPath)
    print("\n")
    print("Place your modified audio files in this location prior to running compile.py.")
    print("The folder structure MUST be maintained! For example, for any Source converted")
    print("file, place the Destination modified file as follows:")
    print("\n")
    print("If you previously extracted an archive or converted a loose folder...")
    print("---")
    print("Source: " + inputPath + "\\my\\file.ogg")
    print("Destination: " + outputPath + "\\my\\file.ogg")
    print("\n")
    print("If you previously extracted the entire game...")
    print("---")
    print("Source: " + inputPath + "\\" + ROOT_MAIN + "\\gameX\\my\\files.ogg")
    print("Destination: " + outputPath + "\\" + ROOT_MAIN + "\\my\\files.ogg")
    print("(substitute " + ROOT_MAIN + "\\ with a DLC folder as needed)")
    print("(yes, you need to OMIT the gameX\\ folder when placing your modified file)")
    print("\n")
    print("When you have placed all your modified files in the created folders, you can")
    print("compile them all at once for your game by running compile.py. It's that simple!")
    print("================================================================================")
    print("\n")


'''
Creates empty folders underneath output for each entry in listHeader.
Called only by start().
'''
def createModDirs(output, listHeader):
    rootOut = os.path.abspath(output)
    for dir in listHeader:
        filepath = os.path.join(rootOut, dir)
        if not os.path.exists(filepath):
            os.makedirs(filepath)


'''
Loads all header file directories for a game folder, and returns the resulting list.
Called only by start().
'''
def loadHeaderDirsGame(input):
    listHeader = []
    listHeader = loadHeaderDirs(input + "\\" + ROOT_MAIN, listHeader)
    listHeader = loadHeaderDirs(input + "\\" + ROOT_SEA, listHeader)
    listHeader = loadHeaderDirs(input + "\\" + ROOT_MECH, listHeader)
    listHeader = loadHeaderDirs(input + "\\" + ROOT_AIR, listHeader)
    return listHeader


'''
Loads the requested header file's directories into a list.
Called only by start().
'''
def loadHeaderDirsFolder(input):
    listHeader = []
    listHeader = loadHeaderDirs(input, listHeader)
    return listHeader


'''
Loads all header file directories from the requested folder into the list.
It is required that listHeader is NOT passed as None.
'''
def loadHeaderDirs(input, listHeader):
    for f in os.listdir(input):
        filepath = os.path.join(input, f)
        try:
            if os.path.isfile(filepath) and filepath.endswith(EXT_HEADER):
                listHeader = loadHeaderDir(filepath, listHeader)
        except Exception as e:
            print("\n")
            print(e)
            print("\nERROR: could not load header " + filepath)
            print("\n")
            quit()

    return listHeader


'''
Loads a single header file into the list.
It is required that listHeader is NOT passed as None.
'''
def loadHeaderDir(input, listHeader):
    with open(input, "r") as infile:
        for line in infile:
            result = line.split(HEADER_DELIM)
            splitext = result[0].rsplit("\\", 1)
            if not splitext[0] in listHeader:
                listHeader.append(splitext[0])

    return listHeader


if __name__ == '__main__':
    processArgs()
