import sys
sys.dont_write_bytecode = True

import os
import subprocess
import shutil
import binascii
from multiprocessing import Process, Lock, Manager
import ast


DUPLICATE_KEYS_EXCLUDE = ["main\\__UNKNOWN\\fsb5\\C0C3EA0A"]


OPT_HELP = ["-h", "--help"]
OPT_LOC = ["-l", "--header_location"]
OPT_OVERWRITE = ["-o", "--overwrite"]
OPT_THREADS = ["-t", "--threads"]


ROOT_MAIN = "main"
ROOT_SEA = "bavarium_sea_heist"
ROOT_MECH = "mech_land_assault"
ROOT_AIR = "sky_fortress"


OUTPUT_MAIN = "dropzone"
OUTPUT_SEA = "dropzone_sea_heist"
OUTPUT_MECH = "dropzone_mech_dlc"
OUTPUT_AIR = "dropzone_sky_fortress"


HEADER_SIZE = 16
HEADER_DELIM = "_____"


LIB_PATH = "lib"
RUN_EXTRACT = "runExtract.bat"
RUN_TO_FSB = "runToFSB.bat"


EXT_FSB = ".fsb5"
EXT_WAV = ".wavc"
EXT_OGG = ".ogg"
EXT_HEADER = "header.dat"


MODE_NONE = -1
MODE_SINGLE = 0
MODE_FOLDER = 1
MODE_GAME = 2


FILE_HEADER = "_" + EXT_HEADER
PATH_CONVERTED = "converted"
PATH_MOD = "__placeChangesHere"
PATH_COMPILED = "__compiled"


'''
Entry point for the script.
Parses all script arguments and runs the compilation process.
Aborts early if any errors are detected (nonexistent files, illegal options, and so on).
'''
def processArgs():
    if len(sys.argv) < 2:
        printHelp()
        return

    mode = MODE_NONE
    alternateHeaders = None
    overwrite = False
    maxThreads = 8
    input = None
    output = None

    i = 1
    while i < len(sys.argv):
        if sys.argv[i] == OPT_HELP[0] or sys.argv[i] == OPT_HELP[1]:
            printHelp()
            return
        elif input == None and (sys.argv[i] == OPT_LOC[0] or sys.argv[i] == OPT_LOC[1]):
            if i + 1 < len(sys.argv):
                alternateHeaders = sys.argv[i + 1]
                i += 1
            else:
                print("expected header path at this point")
        elif input == None and (sys.argv[i] == OPT_OVERWRITE[0] or sys.argv[i] == OPT_OVERWRITE[1]):
            overwrite = True
        elif input == None and (sys.argv[i] == OPT_THREADS[0] or sys.argv[i] == OPT_THREADS[1]):
            if i + 1 < len(sys.argv) and sys.argv[i + 1].isdigit():
                maxThreads = max(int(sys.argv[i + 1]), 1)
                i += 1
            else:
                print("illegal thread count; this option will be ignored")
        elif input == None:
            input = sys.argv[i]
        elif output == None:
            output = sys.argv[i]
        else:
            print("\n")
            print("ERROR: did not expect '" + sys.argv[i] + "' at this time.")
            print("\n")
            print("Run with " + OPT_HELP[0] + " or " + OPT_HELP[1] + " to see available options.")
            return

        i += 1

    if input == None:
        print("\n")
        print("ERROR: no input file or folder was given")
        print("\n")
        print("Run with " + OPT_HELP[0] + " or " + OPT_HELP[1] + " to see available options.")
        return

    if os.path.exists(input):
        if os.path.isdir(input):
            mode = MODE_FOLDER # can correct this later
        else:
            mode = MODE_SINGLE
    # default assume a single file

    if mode == MODE_SINGLE:
        if not input.lower().endswith(EXT_OGG):
            print("\n")
            print("ERROR: '" + input + "' is not a supported file.")
            print("\n")
            print("The supported file type is " + EXT_OGG + ".")
            return
        elif not os.path.exists(input) or os.path.isdir(input):
            print("\n")
            print("ERROR: '" + input + "' is a folder or does not exist.")
            print("\n")
            print("Please make sure you have access to this path, then try again. If your path")
            print("contains a space, the entire path must be enclosed in quotes. A single file is")
            print("required; not a folder.")
            return

        if output is not None and os.path.exists(output) and not os.path.isdir(output):
            print("\n")
            print("ERROR: '" + output + "' is not a folder.")
            print("\n")
            print("When compiling single files, the output (if any) is not permitted to be a file.")
            print("The name of the compiled file will always be automatically determined.")
            print("\n")
            print("Please choose a folder name or remove the output argument, then try again.")
            return

    elif not os.path.exists(input) or not os.path.isdir(input):
        print("\n")
        print("ERROR: '" + input + "' is not a directory or does not exist.")
        print("\n")
        print("Please make sure you have access to this path, then try again.")
        print("\n")
        print("If your path contains a space, the entire path must be enclosed in quotes. If")
        print("your path is not fully-qualified e.g. with C:\\, it must be relative to the")
        print("current working directory (this may NOT be the same as the script directory).")
        return

    if output == None and mode != MODE_SINGLE:
        output = os.path.abspath(os.getcwd()) + "\\" + PATH_COMPILED

    if output != None and os.path.exists(output) and not os.path.isdir(output) \
            and mode != MODE_SINGLE:
        print("\n")
        print("ERROR: output '" + outputDir + "' is a file.")
        print("\n")
        print("The output must be a folder if the input is a folder. Otherwise, the output can")
        print("only be a file if the input is a " + EXT_FSB + " or " + EXT_WAV + ".")
        print("\n")
        print("Please choose a folder or change the script options as needed, then try again.")
        return

    if alternateHeaders is None and mode != MODE_SINGLE \
            and (not os.path.exists(os.path.join(input, PATH_CONVERTED))
            and not os.path.exists(os.path.join(input, PATH_MOD))):
        print("\n")
        print("ERROR: no header path was given, and headers were not found in:\n    " + output)
        print("\n")
        print("If you created mod folders within extract.py, please make sure that your")
        print("input path is set to the same location as the output folder you chose for")
        print("extract.py.")
        print("\n")
        print("If you did not create mod folders, or placed your mod folders in another")
        print("location, the " + OPT_LOC[0] + " option is REQUIRED when compiling any folder.")
        print("\n")
        print("Please correct the script options as needed, then try again.")
        return

    if alternateHeaders is not None:
        if not os.path.exists(alternateHeaders):
            print("\n")
            print("ERROR: headers '" + alternateHeaders + "' do not exist.")
            print("\n")
            print("Please choose an existing folder or file, then try again.")
            return

        elif os.path.isdir(alternateHeaders) and mode == MODE_SINGLE:
            print("\n")
            print("ERROR: headers '" + alternateHeaders + "' are a folder.")
            print("\n")
            print("When compiling a single file, the alternate headers must also be a single file.")
            print("Please choose a valid file, then try again.")
            return

        elif not os.path.isdir(alternateHeaders) and mode != MODE_SINGLE:
            print("\n")
            print("ERROR: headers '" + alternateHeaders + "' is a file.")
            print("\n")
            print("When compiling a folder, the alternate headers must also be a folder. Please")
            print("choose a valid folder, then try again.")
            return

    if alternateHeaders is None and mode != MODE_SINGLE:
        if not os.path.exists(os.path.join(input, PATH_MOD)):
            print("\n")
            print("ERROR: expected " +  PATH_MOD + " folder in input '" + input + "'")
            print("\n")
            print("Please make sure this folder exists, then try again.")
            return
        alternateHeaders = input
        input = os.path.join(input, PATH_MOD)

    if mode != MODE_SINGLE:
        mode = getModeFromFolder(alternateHeaders)

    start(input, output, mode, overwrite, alternateHeaders, maxThreads)


'''
Displays the usage information for this script.
Called only by processArgs().
'''
def printHelp():
    print("\n")
    print("================================================================================")
    print("Lord Zapharos Audio Compiler for Just Cause 3")
    print("================================================================================")
    print("\n")
    print("This utility allows you to repack any modified audio files back into your game,")
    print("or simply convert them for distribution to others. You must have previously run")
    print("extract.py BEFORE using this tool!")
    print("\n")
    print("Please note that fmod banks CANNOT be compiled by this tool. You must use the")
    print("Fmod Bank Tools in the " + LIB_PATH + " folder to compile those files.")
    print("\n")
    print("USAGE: compile.py [OPTIONS] path\\to\\modified\\file\\or\\files [outputFolder]")
    print("\n")
    print("If an output folder is not given, all compiled audio files will be placed in the")
    print("current working directory in a subfolder called '" + PATH_COMPILED + \
            "'. If a full game")
    print("folder is given, the appropriate " + OUTPUT_MAIN + \
            "\\ folders will be automatically created")
    print("created within the output folder. For non-game folders, the output folder")
    print("structure will mirror the input.")
    print("\n")
    print("If an output folder is not given when compiling single files, the compiled file")
    print("will be placed next to the input file. Specifying an output FOLDER will change")
    print("this behavior. It is not possible to output to a different file name, as this")
    print("would not permit reintegration into the game.")
    print("\n")
    print("When compiling any folder, the path\\to\\modified\\file\\or\\files must be structured")
    print("in a special way (Just Cause 3 is very sensitive about file paths). If you did")
    print("not create mod folders when running extract.py, you can run createModFolders.py")
    print("at any time to create them. Make sure your files have been placed correctly")
    print("BEFORE running this tool!")
    print("\n")
    print("OPTIONS:")
    print("\n")
    print(OPT_HELP[0] + ", " + OPT_HELP[1] + \
            "              shows this message and exits.")
    print("\n")
    print(OPT_OVERWRITE[0] + ", " + OPT_OVERWRITE[1] +  \
            "         overwrites any files at the destination. When compiling")
    print("                        a folder, omitting this option will cause only new files")
    print("                        (i.e. not already at the destination) to be compiled.")
    print("\n")
    print(OPT_LOC[0] + ", " + OPT_LOC[1] + \
            "   specifies an alternate location for header files. If you")
    print("                        created mod folders when extracting an archive or game,")
    print("                        this argument is optional. If you did NOT place your mod")
    print("                        mod folders in the same location as the extracted files,")
    print("                        this option is REQUIRED.")
    print("\n")
    print(OPT_THREADS[0] + ", " + OPT_THREADS[1] + \
            "           valid only with a folder; sets how many threads should")
    print("                        be used when compiling audio files. A larger value will")
    print("                        increase conversion speed, but demands more CPU power. A")
    print("                        value between 2-16 is recommended; the default is 8. A")
    print("                        good rule of thumb is twice your CPU's core count.")
    print("\n")
    print("EXAMPLES:")
    print("\n")
    print("compile.py -l other\\header.dat path\\to\\file.ogg     compiles file.ogg using the")
    print("                                                    header file at other\\, with")
    print("                                                    the file placed in the same")
    print("                                                    location as file.ogg.")
    print("\n")
    print("compile.py path\\to\\extracted\\archive outputDir      compiles a prior extracted")
    print("                                                    archive, placing the results")
    print("                                                    in outputDir\\.")
    print("\n")
    print("compile.py -t 16 path\\to\\extracted\\game             compiles a prior extracted")
    print("                                                    game using 16 threads.")
    print("\n")
    print("compile.py -l path\\to\\extracted path\\to\\mod         compiles a prior extracted")
    print("                                                    game in which the output was")
    print("                                                    path\\to\\extracted, but the")
    print("                                                    folder with modified files")
    print("                                                    is located at path\\to\\mod.")
    print("\n")
    print("================================================================================")
    print("\n")


'''
Automatically deduces the operating mode (if not a single file).
Called only by processArgs().
'''
def getModeFromFolder(alternateHeaders):
    inputHeader = os.path.join(alternateHeaders, EXT_HEADER)
    if not os.path.exists(inputHeader):
        inputHeader = os.path.join(alternateHeaders, PATH_CONVERTED)
        inputHeader = os.path.join(inputHeader, ROOT_MAIN)
        if os.path.exists(inputHeader):
            return MODE_GAME
        else:
            return MODE_NONE
    else:
        return MODE_FOLDER


'''
Initiates the compilation process.
Called only by processArgs().
'''
def start(input, output, mode, overwrite, alternateHeaders, maxThreads):
    print("\n")
    if mode == MODE_SINGLE:
        compileSingleNonFolder(input, output, overwrite, alternateHeaders)
        return

    if mode == MODE_GAME:
        compileGame(input, output, overwrite, alternateHeaders, maxThreads)
    else:
        compileFolder(input, output, overwrite, alternateHeaders, maxThreads)


'''
Compiles a single non-foldered file.
Called only by start().
'''
def compileSingleNonFolder(input, output, overwrite, alternateHeaders):
    headerFile = alternateHeaders
    if headerFile is None:
        headerFile = input[:-len(EXT_OGG)] + FILE_HEADER

    if not os.path.exists(headerFile) or os.path.isdir(headerFile):
        print("ERROR: the header file '" + headerFile + "' is a folder or does not exist.")
        print("\n")
        print("Please specify a valid header file via " + OPT_LOC[0] + ", then try again.")
        quit()

    dictHeader, dictOut = loadHeader(headerFile, None, None)
    if len(dictHeader) != 1:
        print("ERROR: the header file '" + headerFile + "' is empty or corrupted.")
        print("\n")
        print("Please specify a valid header file via " + OPT_LOC[0] + ", then try again.")
        quit()

    for key in dictOut: # guaranteed to run exactly once by previous condition
        outputFile = key + "." + dictOut[key]
    for key in dictHeader:
        header = dictHeader[key]

    if output != None:
        if not os.path.exists(output):
            os.makedirs(output)
        output = os.path.join(output, outputFile)
    else:
        output = outputFile

    if os.path.exists(output):
        if os.path.isdir(output):
            print("ERROR: the destination file '" + output + "' is a folder.")
            print("\n")
            print("Please rename, move, or delete this folder, then try again.")
            quit()
        elif not overwrite:
            print("ERROR: the destination file '" + output + "' already exists.")
            print("\n")
            print("Please rename, move, or delete this file, then try again. To overwrite the file,")
            print("set " + OPT_OVERWRITE[0] + " in your script options.")
            quit()

    compile(input, output, header)


'''
Compiles all files for a game.
Called only by start().
'''
def compileGame(input, output, overwrite, alternateHeaders, maxThreads):
    inputHeader = os.path.join(alternateHeaders, PATH_CONVERTED)
    if not os.path.exists(inputHeader):
        print("ERROR: could not find " + inputHeader)
        print("\n")
        print("Please verify that your header or input folder is correct, then try again.")
        quit()

    dictHeader, dictOut = loadHeadersGame(inputHeader)

    relReplace1 = [ROOT_MAIN, ROOT_SEA, ROOT_MECH, ROOT_AIR]
    relReplace2 = [OUTPUT_MAIN, OUTPUT_SEA, OUTPUT_MECH, OUTPUT_AIR]

    compileTreeExecutor(input, output, overwrite, maxThreads, dictHeader, dictOut, relReplace1, \
            relReplace2)


'''
Compiles all loose files in a folder.
Called only by start().
'''
def compileFolder(input, output, overwrite, alternateHeaders, maxThreads):
    inputHeader = os.path.join(alternateHeaders, EXT_HEADER)
    if not os.path.exists(inputHeader):
        print("ERROR: could not find " + EXT_HEADER + " in " + alternateHeaders)
        print("\n")
        print("Please verify that your header or input folder is correct, then try again.")
        quit()

    dictHeader, dictOut = loadHeadersFolder(inputHeader)
    compileTreeExecutor(input, output, overwrite, maxThreads, dictHeader, dictOut, None, None)


'''
Performs a threaded compilation of files from their playable format into a game format.
Called by compileGame() or by compileFolder().
'''
def compileTreeExecutor(input, output, overwrite, maxThreads, dictHeader, dictOut, relReplace1, \
        relReplace2):
    manager = Manager()
    pool = []

    compileTree("", input, output, overwrite, pool, maxThreads, dictHeader, dictOut, relReplace1, \
            relReplace2)
    for p in pool:
        p.join()


'''
Recursively examines each file in the provided input and performs a (threaded) compilation.
Called only by compileTreeExecutor().
'''
def compileTree(relBase, input, output, overwrite, pool, maxThreads, dictHeader, dictOut, \
        relReplace1, relReplace2):
    input = os.path.abspath(input)

    for f in os.listdir(input):
        filepath = os.path.join(input, f)
        try:
            if (os.path.isfile(filepath) or os.path.islink(filepath)) \
                    and filepath.endswith(EXT_OGG):
                rawName = os.path.basename(filepath[:-len(EXT_OGG)])
                key = relBase + "\\" + rawName

                if key in dictHeader:
                    outputName = rawName + "." + dictOut[key]
                    if relReplace1 != None:
                        for i, item in enumerate(relReplace1):
                            if relBase.startswith(item):
                                relBaseRepl = relReplace2[i] + relBase[len(item):]

                    outputFullPath = output + "\\" + relBaseRepl + "\\" + outputName

                    if os.path.exists(outputFullPath):
                        if not overwrite or os.path.isdir(outputFullPath):
                            return

                    p = Process(target=compileSingleFromTree, args=(filepath, output, \
                            relBaseRepl, outputFullPath, dictHeader[key]))
                    pool.append(p)
                    p.start()

                    if len(pool) >= maxThreads:
                        for p in pool:
                            p.join()
                        pool.clear()

            elif os.path.isdir(filepath):
                if relBase == "":
                    relBaseNext = f
                else:
                    relBaseNext = relBase + "\\" + f

                compileTree(relBaseNext, filepath, output, overwrite, pool, maxThreads, \
                        dictHeader, dictOut, relReplace1, relReplace2)

        except Exception as e:
            print("\n")
            print(e)
            print("\nERROR: could not iterate " + input)
            print("\n")
            quit()


'''
Compiles a single file from compileTree() into its game format.
'''
def compileSingleFromTree(input, outputRoot, outputPrefix, outputFullPath, outputHeader):
    outputDirs = os.path.join(outputRoot, outputPrefix)
    try:
        if not os.path.exists(outputDirs):
            os.makedirs(outputDirs)
    except Exception:
        pass # possible race condition when creating parent folders (shared between files)

    compile(input, outputFullPath, outputHeader)


'''
Compiles a single file and afterwards prepends its header.
'''
def compile(input, output, header):
    if os.path.exists(output):
        os.remove(output)

    try:
        command = RUN_TO_FSB + " \"" + os.path.abspath(input) + "\" \"" + \
                os.path.abspath(output) + "\""
        subprocess.check_call(command, shell=True, cwd=LIB_PATH)

        with open(output, "rb") as infile:
            data = infile.read()

        data = binascii.unhexlify(header) + data
        with open(output, "wb") as outfile:
            outfile.write(data)
    except Exception as e:
        print("\n")
        print(e)
        print("\nERROR: could not compile audio stream for " + input)
        print("\n")
        quit()


'''
Loads all header files for a game folder, and returns the resulting recompilation dictionaries.
Called only by start().
'''
def loadHeadersGame(input):
    dictHeader = {}
    dictOut = {}
    dictHeader, dictOut = loadHeaders(input + "\\" + ROOT_MAIN, dictHeader, dictOut)
    dictHeader, dictOut = loadHeaders(input + "\\" + ROOT_SEA, dictHeader, dictOut)
    dictHeader, dictOut = loadHeaders(input + "\\" + ROOT_MECH, dictHeader, dictOut)
    dictHeader, dictOut = loadHeaders(input + "\\" + ROOT_AIR, dictHeader, dictOut)
    return dictHeader, dictOut


'''
Loads the requested header file into a set of recompilation dictionaries.
Called only by start().
'''
def loadHeadersFolder(input):
    dictHeader = {}
    dictOut = {}
    dictHeader, dictOut = loadHeader(input, dictHeader, dictOut)
    return dictHeader, dictOut


'''
Loads all header files from the requested folder into the dictionary.
It is required that dictHeader and dictOut are NOT passed as None.
'''
def loadHeaders(input, dictHeader, dictOut):
    for f in os.listdir(input):
        filepath = os.path.join(input, f)
        try:
            if os.path.isfile(filepath) and filepath.endswith(EXT_HEADER):
                dictHeader, dictOut = loadHeader(filepath, dictHeader, dictOut)
        except Exception as e:
            print("\n")
            print(e)
            print("\nERROR: could not load header " + filepath)
            print("\n")
            quit()

    return dictHeader, dictOut


'''
Loads a single header file into the given dictionary.
The dictHeader and dictOut will be created if they are currently set to None.
'''
def loadHeader(input, dictHeader, dictOut):
    if dictHeader == None:
        dictHeader = {}
    if dictOut == None:
        dictOut = {}

    with open(input, "r") as infile:
        for line in infile:
            try:
                result = line.split(HEADER_DELIM)
                splitext = result[0].rsplit(".", 1)

                if splitext[0] in dictHeader:
                    print("A duplicate key '" + splitext[0] + "'")
                    print("was found in: " + input)
                    print("\n")
                    print("Duplicate keys prevent the converter " + \
                            "from correctly determining how to compile")
                    print("your file(s). Please make sure you have selected the correct header file")
                    print("location, then try again.")
                    print("\n")
                    quit()

                if splitext[0] in DUPLICATE_KEYS_EXCLUDE:
                    print("Skipping files with '" + splitext[0] + "' (known duplicate/ambiguous)")
                else:
                    dictHeader[splitext[0]] = ast.literal_eval(result[1])
                    dictOut[splitext[0]] = splitext[1]
            except Exception:
                pass

    return dictHeader, dictOut



if __name__ == '__main__':
    processArgs()
